// Pull out the full ISA process sequence (incl. all metadata) leading to the first Raw Data Node 

// Dependencies

#r "nuget: ARCtrl"
#r "nuget: ARCtrl.QueryModel"
#r "nuget: FsSpreadsheet.CsvIO"

open ARCtrl
open ARCtrl.QueryModel
open ARCtrl.Helper
open FsSpreadsheet.Net
open FsSpreadsheet.CsvIO


// test parameters
// let source = __SOURCE_DIRECTORY__
// let arcPath = System.IO.Path.Combine(source, "../../")
// let assayName = "RNASeq"
// let startingNodeNum = 0
// let outName = "rnaseq-samples"


// input parameters

let args : string array = fsi.CommandLineArgs |> Array.tail
let arcPath = args.[0]
let assayName = args.[1]
let startingNodeNum = args.[2] |> int
let outName = args.[3]


type ArcTables with

    member this.IgnoreShitty() : ArcTables = 
        this.Tables
        |> ResizeArray.filter (fun t ->
            t.TryGetInputColumn().IsSome && t.TryGetOutputColumn().IsSome       
        )
        |> ArcTables

// Load ARC

// Remove all tables with either an input or output column missing
let clean (a : ARC)  = 
    a.ISA.Value.Assays |> Seq.iter (fun a -> 
        a.Tables
        |> Seq.toArray
        |> Seq.iter (fun t -> 
            if  not (t.TryGetInputColumn().IsSome && t.TryGetOutputColumn().IsSome) then
                a.RemoveTable t.Name
        
        )
    
    )
    a.ISA.Value.Studies |> Seq.iter (fun s -> 
        s.Tables
        |> Seq.toArray
        |> Seq.iter (fun t -> 
            if  not (t.TryGetInputColumn().IsSome && t.TryGetOutputColumn().IsSome) then
                s.RemoveTable t.Name
        
        )
    
    )
    a

// transform all data cells to freetext cells
let shittify (a : ARC) = 
    a.ISA.Value.Assays |> Seq.iter (fun a -> 
        a.Tables
        |> Seq.toArray
        |> Seq.iter (fun t -> 
            t.Values
            |> Seq.toArray
            |> Seq.iter (fun kv ->
                if kv.Value.isData then 
                    t.Values.[kv.Key] <- CompositeCell.createFreeText(kv.Value.AsData.NameText)         
            )
        )
    
    )
    a.ISA.Value.Studies |> Seq.iter (fun s -> 
        s.Tables
        |> Seq.toArray
        |> Seq.iter (fun t -> 
            t.Values
            |> Seq.toArray
            |> Seq.iter (fun kv ->
                if kv.Value.isData then 
                    t.Values.[kv.Key] <- CompositeCell.createFreeText(kv.Value.AsData.NameText)         
            )
        )
    )
    a


let arc = ARC.load(arcPath) |> clean |> shittify


let inv = arc.ISA.Value 

// Load first data node

let firstData = inv.GetAssay(assayName).FirstData


// Create headers for output table
let headers = [
    
    CompositeHeader.Input IOType.Source
    
    // CompositeHeader.Input IOType.Sample

    for v in inv.ArcTables.IgnoreShitty().ValuesOf firstData.[0].Name do
        if v.IsCharacteristicValue then
            CompositeHeader.Characteristic v.Category
        elif v.IsParameterValue then
            CompositeHeader.Parameter v.Category
        elif v.IsFactorValue then
            CompositeHeader.Factor v.Category
        elif v.IsComponent then
            CompositeHeader.Component v.Category
        else failwithf "what the f is %O" v

    CompositeHeader.Output IOType.Data
]


// Create rows

let getRow (d: QNode) = 
    [|
    CompositeCell.createFreeText (inv.ArcTables.SourcesOf d).[startingNodeNum].Name

    for v in inv.ArcTables.ValuesOf d do
        if v.HasUnit then
            CompositeCell.Unitized(v.ValueText, v.Unit)
        else
            CompositeCell.Term(v.Value.AsOntology())

    CompositeCell.Data (Data(name = d.Name))
    
    |]

// Combine into table

let t = ArcTable.init "FullTable"
t.Headers <- ResizeArray headers


let d = firstData.Head

inv.ArcTables.SamplesOf d


for d in firstData do
    t.AddRow (getRow d)

// Small detour via workbook
let ws = Spreadsheet.ArcTable.toFsWorksheet None t

let wb = new FsSpreadsheet.FsWorkbook()

wb.AddWorksheet ws

// Write to xlsx
wb.ToXlsxFile ($"{outName}.xlsx")

// Write to csv
wb.ToCsvFile ($"{outName}.csv")
